class CalendarApp {
    constructor() {
        this.currentDate = new Date();
        this.currentMonth = this.currentDate.getMonth();
        this.currentYear = this.currentDate.getFullYear();
        this.selectedDate = null;
        this.entries = this.loadEntries();
        this.editingEntry = null;
        this.selectedColor = '#3b82f6';
        this.bulkSelectedColor = '#3b82f6';
        this.titleTemplates = this.loadTitleTemplates();
        this.selectedWeekdays = new Set();

        this.init();
    }

    init() {
        this.renderCalendar();
        this.attachEventListeners();
        this.setupColorPicker();
        this.setupBulkColorPicker();
        this.setupTitleAutocomplete();
    }

    attachEventListeners() {
        document.getElementById('prevMonth').addEventListener('click', () => this.changeMonth(-1));
        document.getElementById('nextMonth').addEventListener('click', () => this.changeMonth(1));
        document.getElementById('todayBtn').addEventListener('click', () => this.goToToday());

        document.getElementById('closeModal').addEventListener('click', () => this.closeModal());
        document.getElementById('cancelBtn').addEventListener('click', () => this.closeModal());
        document.getElementById('saveBtn').addEventListener('click', () => this.saveEntry());

        document.getElementById('closeDayPanel').addEventListener('click', () => this.closeDayPanel());
        document.getElementById('addEntryBtn').addEventListener('click', () => this.openModal());
        document.getElementById('bulkEntryBtn').addEventListener('click', () => this.openBulkModal());

        document.getElementById('closeBulkModal').addEventListener('click', () => this.closeBulkModal());
        document.getElementById('cancelBulkBtn').addEventListener('click', () => this.closeBulkModal());
        document.getElementById('saveBulkBtn').addEventListener('click', () => this.saveBulkEntry());

        document.getElementById('entryModal').addEventListener('click', (e) => {
            if (e.target.id === 'entryModal') this.closeModal();
        });

        document.getElementById('bulkEntryModal').addEventListener('click', (e) => {
            if (e.target.id === 'bulkEntryModal') this.closeBulkModal();
        });

        document.querySelectorAll('.weekday-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const day = parseInt(btn.dataset.day);
                if (this.selectedWeekdays.has(day)) {
                    this.selectedWeekdays.delete(day);
                    btn.classList.remove('active');
                } else {
                    this.selectedWeekdays.add(day);
                    btn.classList.add('active');
                }
            });
        });

        window.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeModal();
                this.closeDayPanel();
                this.closeBulkModal();
            }
        });
    }

    setupColorPicker() {
        const colorPresets = document.querySelectorAll('#colorPresets .color-preset');
        const colorInput = document.getElementById('entryColor');

        colorPresets.forEach(preset => {
            preset.addEventListener('click', () => {
                const color = preset.dataset.color;
                this.selectedColor = color;
                colorInput.value = color;

                colorPresets.forEach(p => p.classList.remove('selected'));
                preset.classList.add('selected');
            });
        });

        colorInput.addEventListener('input', (e) => {
            this.selectedColor = e.target.value;
            colorPresets.forEach(p => p.classList.remove('selected'));
        });
    }

    setupBulkColorPicker() {
        const colorPresets = document.querySelectorAll('#bulkColorPresets .color-preset');
        const colorInput = document.getElementById('bulkEntryColor');

        colorPresets.forEach(preset => {
            preset.addEventListener('click', () => {
                const color = preset.dataset.color;
                this.bulkSelectedColor = color;
                colorInput.value = color;

                colorPresets.forEach(p => p.classList.remove('selected'));
                preset.classList.add('selected');
            });
        });

        colorInput.addEventListener('input', (e) => {
            this.bulkSelectedColor = e.target.value;
            colorPresets.forEach(p => p.classList.remove('selected'));
        });
    }

    setupTitleAutocomplete() {
        const entryTitle = document.getElementById('entryTitle');
        const bulkEntryTitle = document.getElementById('bulkEntryTitle');

        entryTitle.addEventListener('input', (e) => {
            this.updateTitleSuggestions();
            this.checkTitleTemplate(e.target.value);
        });

        bulkEntryTitle.addEventListener('input', (e) => {
            this.updateTitleSuggestions();
            this.checkBulkTitleTemplate(e.target.value);
        });
    }

    updateTitleSuggestions() {
        const datalist = document.getElementById('titleSuggestions');
        datalist.innerHTML = '';

        Object.keys(this.titleTemplates).forEach(title => {
            const option = document.createElement('option');
            option.value = title;
            datalist.appendChild(option);
        });
    }

    checkTitleTemplate(title) {
        const template = this.titleTemplates[title];
        if (template) {
            document.getElementById('entryDescription').value = template.description || '';
            document.getElementById('entryTime').value = template.time || '';
            document.getElementById('entryCategory').value = template.category || 'work';
            document.getElementById('entryPriority').value = template.priority || 'medium';

            this.selectedColor = template.color || '#3b82f6';
            document.getElementById('entryColor').value = this.selectedColor;
            this.updateColorPresetSelection(this.selectedColor);
        }
    }

    checkBulkTitleTemplate(title) {
        const template = this.titleTemplates[title];
        if (template) {
            document.getElementById('bulkEntryDescription').value = template.description || '';
            document.getElementById('bulkEntryTime').value = template.time || '';
            document.getElementById('bulkEntryCategory').value = template.category || 'work';
            document.getElementById('bulkEntryPriority').value = template.priority || 'medium';

            this.bulkSelectedColor = template.color || '#3b82f6';
            document.getElementById('bulkEntryColor').value = this.bulkSelectedColor;
            this.updateBulkColorPresetSelection(this.bulkSelectedColor);
        }
    }

    renderCalendar() {
        const firstDay = new Date(this.currentYear, this.currentMonth, 1);
        const lastDay = new Date(this.currentYear, this.currentMonth + 1, 0);
        const prevLastDay = new Date(this.currentYear, this.currentMonth, 0);

        const firstDayIndex = firstDay.getDay();
        const lastDayIndex = lastDay.getDay();
        const lastDayDate = lastDay.getDate();
        const prevLastDayDate = prevLastDay.getDate();

        document.getElementById('currentMonth').textContent =
            `${this.getMonthName(this.currentMonth)} ${this.currentYear}`;

        let days = '';

        for (let x = firstDayIndex; x > 0; x--) {
            const date = prevLastDayDate - x + 1;
            const dateStr = this.formatDate(new Date(this.currentYear, this.currentMonth - 1, date));
            days += this.createDayElement(date, dateStr, 'other-month');
        }

        for (let i = 1; i <= lastDayDate; i++) {
            const dateStr = this.formatDate(new Date(this.currentYear, this.currentMonth, i));
            const isToday = this.isToday(i, this.currentMonth, this.currentYear);
            const classes = isToday ? 'today' : '';
            days += this.createDayElement(i, dateStr, classes);
        }

        const nextDays = 7 - lastDayIndex - 1;
        for (let j = 1; j <= nextDays; j++) {
            const dateStr = this.formatDate(new Date(this.currentYear, this.currentMonth + 1, j));
            days += this.createDayElement(j, dateStr, 'other-month');
        }

        document.getElementById('calendarGrid').innerHTML = days;

        document.querySelectorAll('.calendar-day').forEach(day => {
            day.addEventListener('click', (e) => {
                const date = e.currentTarget.dataset.date;
                this.selectDate(date);
            });
        });
    }

    createDayElement(dayNumber, dateStr, classes = '') {
        const dayEntries = this.entries[dateStr] || [];
        const entryCount = dayEntries.length;

        let entryDotsHTML = '';
        if (entryCount > 0) {
            const displayCount = Math.min(entryCount, 10);
            const barHeight = displayCount <= 5 ? 6 : Math.max(3, Math.floor(80 / displayCount));

            entryDotsHTML = dayEntries.slice(0, displayCount).map(entry => {
                const color = entry.color || this.getCategoryColor(entry.category);
                return `<div class="entry-dot" style="background: ${color}; height: ${barHeight}px;"></div>`;
            }).join('');
        }

        const countBadge = entryCount > 0 ? `<div class="entry-count">${entryCount}</div>` : '';

        return `
            <div class="calendar-day ${classes}" data-date="${dateStr}">
                <div class="day-number">${dayNumber}</div>
                <div class="day-entries">
                    ${entryDotsHTML}
                </div>
                ${countBadge}
            </div>
        `;
    }

    selectDate(dateStr) {
        this.selectedDate = dateStr;

        document.querySelectorAll('.calendar-day').forEach(day => {
            day.classList.remove('selected');
        });

        const selectedDay = document.querySelector(`[data-date="${dateStr}"]`);
        if (selectedDay) {
            selectedDay.classList.add('selected');
        }

        this.showDayPanel(dateStr);
    }

    showDayPanel(dateStr) {
        const date = new Date(dateStr);
        const dayName = this.getDayName(date.getDay());
        const monthName = this.getMonthName(date.getMonth());

        document.getElementById('panelDate').textContent =
            `${dayName}, ${monthName} ${date.getDate()}, ${date.getFullYear()}`;

        const entries = this.entries[dateStr] || [];
        const panelBody = document.getElementById('dayPanelBody');

        if (entries.length === 0) {
            panelBody.innerHTML = `
                <div class="empty-state">
                    <p>No entries for this day</p>
                </div>
            `;
        } else {
            const sortedEntries = entries.sort((a, b) => {
                if (a.time && b.time) return a.time.localeCompare(b.time);
                if (a.time) return -1;
                if (b.time) return 1;
                return 0;
            });

            panelBody.innerHTML = sortedEntries.map((entry, index) => this.createEntryCard(entry, index)).join('');

            document.querySelectorAll('.btn-edit').forEach(btn => {
                btn.addEventListener('click', (e) => {
                    const index = parseInt(e.target.dataset.index);
                    this.editEntry(index);
                });
            });

            document.querySelectorAll('.btn-delete').forEach(btn => {
                btn.addEventListener('click', (e) => {
                    const index = parseInt(e.target.dataset.index);
                    this.deleteEntry(index);
                });
            });
        }

        document.getElementById('dayPanel').classList.add('active');
    }

    createEntryCard(entry, index) {
        const timeHTML = entry.time ? `<div class="entry-time">${this.formatTime(entry.time)}</div>` : '';
        const descriptionHTML = entry.description ?
            `<div class="entry-description">${this.escapeHtml(entry.description)}</div>` : '';
        const color = entry.color || this.getCategoryColor(entry.category);

        return `
            <div class="entry-card" style="border-left-color: ${color};">
                <div class="entry-header">
                    <div class="entry-title">${this.escapeHtml(entry.title)}</div>
                    <div class="entry-actions">
                        <button class="btn-icon-small btn-edit" data-index="${index}">✏️</button>
                        <button class="btn-icon-small btn-delete" data-index="${index}">🗑️</button>
                    </div>
                </div>
                ${timeHTML}
                ${descriptionHTML}
                <div class="entry-meta">
                    <div class="entry-badge category">${this.escapeHtml(entry.category)}</div>
                    <div class="entry-badge priority-${this.sanitizeClass(entry.priority)}">${this.escapeHtml(entry.priority)} priority</div>
                </div>
            </div>
        `;
    }

    closeDayPanel() {
        document.getElementById('dayPanel').classList.remove('active');
        document.querySelectorAll('.calendar-day').forEach(day => {
            day.classList.remove('selected');
        });
        this.selectedDate = null;
    }

    openModal() {
        if (!this.selectedDate) {
            alert('Please select a date first');
            return;
        }

        document.getElementById('modalTitle').textContent = this.editingEntry !== null ? 'Edit Entry' : 'Add Entry';
        document.getElementById('entryModal').classList.add('active');

        if (this.editingEntry !== null) {
            const entry = this.entries[this.selectedDate][this.editingEntry];
            document.getElementById('entryTitle').value = entry.title;
            document.getElementById('entryDescription').value = entry.description || '';
            document.getElementById('entryTime').value = entry.time || '';
            document.getElementById('entryCategory').value = entry.category;
            document.getElementById('entryPriority').value = entry.priority;

            const color = entry.color || this.getCategoryColor(entry.category);
            this.selectedColor = color;
            document.getElementById('entryColor').value = color;
            this.updateColorPresetSelection(color);
        } else {
            document.getElementById('entryTitle').value = '';
            document.getElementById('entryDescription').value = '';
            document.getElementById('entryTime').value = '';
            document.getElementById('entryCategory').value = 'work';
            document.getElementById('entryPriority').value = 'medium';

            this.selectedColor = '#3b82f6';
            document.getElementById('entryColor').value = '#3b82f6';
            this.updateColorPresetSelection('#3b82f6');
        }

        setTimeout(() => document.getElementById('entryTitle').focus(), 100);
    }

    closeModal() {
        document.getElementById('entryModal').classList.remove('active');
        this.editingEntry = null;
    }

    openBulkModal() {
        document.getElementById('bulkEntryModal').classList.add('active');

        document.getElementById('bulkEntryTitle').value = '';
        document.getElementById('bulkEntryDescription').value = '';
        document.getElementById('bulkEntryTime').value = '';
        document.getElementById('bulkEntryCategory').value = 'work';
        document.getElementById('bulkEntryPriority').value = 'medium';

        this.bulkSelectedColor = '#3b82f6';
        document.getElementById('bulkEntryColor').value = '#3b82f6';
        this.updateBulkColorPresetSelection('#3b82f6');

        const today = new Date();
        document.getElementById('bulkStartDate').value = this.formatDate(today);
        document.getElementById('bulkEndDate').value = this.formatDate(new Date(today.getTime() + 7 * 24 * 60 * 60 * 1000));

        this.selectedWeekdays.clear();
        document.querySelectorAll('.weekday-btn').forEach(btn => btn.classList.remove('active'));

        setTimeout(() => document.getElementById('bulkEntryTitle').focus(), 100);
    }

    closeBulkModal() {
        document.getElementById('bulkEntryModal').classList.remove('active');
    }

    saveBulkEntry() {
        const title = document.getElementById('bulkEntryTitle').value.trim();
        const startDate = document.getElementById('bulkStartDate').value;
        const endDate = document.getElementById('bulkEndDate').value;

        if (!title) {
            alert('Please enter a title');
            return;
        }

        if (!startDate || !endDate) {
            alert('Please select start and end dates');
            return;
        }

        const entry = {
            title: title,
            description: document.getElementById('bulkEntryDescription').value.trim(),
            time: document.getElementById('bulkEntryTime').value,
            category: document.getElementById('bulkEntryCategory').value,
            priority: document.getElementById('bulkEntryPriority').value,
            color: this.bulkSelectedColor,
            createdAt: new Date().toISOString()
        };

        this.saveTitleTemplate(entry);

        const start = new Date(startDate);
        const end = new Date(endDate);
        let createdCount = 0;

        for (let date = new Date(start); date <= end; date.setDate(date.getDate() + 1)) {
            if (this.selectedWeekdays.size === 0 || this.selectedWeekdays.has(date.getDay())) {
                const dateStr = this.formatDate(date);

                if (!this.entries[dateStr]) {
                    this.entries[dateStr] = [];
                }

                this.entries[dateStr].push({...entry});
                createdCount++;
            }
        }

        this.saveEntries();
        this.closeBulkModal();
        this.renderCalendar();

        if (this.selectedDate) {
            this.showDayPanel(this.selectedDate);
        }

        alert(`Created ${createdCount} entries successfully!`);
    }

    saveEntry() {
        const title = document.getElementById('entryTitle').value.trim();

        if (!title) {
            alert('Please enter a title');
            return;
        }

        const entry = {
            title: title,
            description: document.getElementById('entryDescription').value.trim(),
            time: document.getElementById('entryTime').value,
            category: document.getElementById('entryCategory').value,
            priority: document.getElementById('entryPriority').value,
            color: this.selectedColor,
            createdAt: new Date().toISOString()
        };

        this.saveTitleTemplate(entry);

        if (!this.entries[this.selectedDate]) {
            this.entries[this.selectedDate] = [];
        }

        if (this.editingEntry !== null) {
            this.entries[this.selectedDate][this.editingEntry] = entry;
        } else {
            this.entries[this.selectedDate].push(entry);
        }

        this.saveEntries();
        this.closeModal();
        this.renderCalendar();
        this.showDayPanel(this.selectedDate);
    }

    editEntry(index) {
        this.editingEntry = index;
        this.openModal();
    }

    deleteEntry(index) {
        if (!confirm('Are you sure you want to delete this entry?')) {
            return;
        }

        this.entries[this.selectedDate].splice(index, 1);

        if (this.entries[this.selectedDate].length === 0) {
            delete this.entries[this.selectedDate];
        }

        this.saveEntries();
        this.renderCalendar();
        this.showDayPanel(this.selectedDate);
    }

    changeMonth(direction) {
        this.currentMonth += direction;

        if (this.currentMonth > 11) {
            this.currentMonth = 0;
            this.currentYear++;
        } else if (this.currentMonth < 0) {
            this.currentMonth = 11;
            this.currentYear--;
        }

        this.renderCalendar();
        this.closeDayPanel();
    }

    goToToday() {
        const today = new Date();
        this.currentMonth = today.getMonth();
        this.currentYear = today.getFullYear();
        this.renderCalendar();

        const todayStr = this.formatDate(today);
        this.selectDate(todayStr);
    }

    isToday(day, month, year) {
        const today = new Date();
        return day === today.getDate() &&
               month === today.getMonth() &&
               year === today.getFullYear();
    }

    formatDate(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    formatTime(time) {
        if (!time) return '';
        const [hours, minutes] = time.split(':');
        const hour = parseInt(hours);
        const ampm = hour >= 12 ? 'PM' : 'AM';
        const displayHour = hour % 12 || 12;
        return `${displayHour}:${minutes} ${ampm}`;
    }

    getMonthName(monthIndex) {
        const months = ['January', 'February', 'March', 'April', 'May', 'June',
                       'July', 'August', 'September', 'October', 'November', 'December'];
        return months[monthIndex];
    }

    getDayName(dayIndex) {
        const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        return days[dayIndex];
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    sanitizeClass(className) {
        const allowedClasses = ['work', 'personal', 'health', 'social', 'finance', 'other', 'low', 'medium', 'high'];
        return allowedClasses.includes(className) ? className : 'other';
    }

    getCategoryColor(category) {
        const colors = {
            work: '#3b82f6',
            personal: '#8b5cf6',
            health: '#10b981',
            social: '#f59e0b',
            finance: '#ef4444',
            other: '#6b7280'
        };
        return colors[category] || colors.other;
    }

    updateColorPresetSelection(color) {
        const colorPresets = document.querySelectorAll('#colorPresets .color-preset');
        colorPresets.forEach(preset => {
            if (preset.dataset.color === color) {
                preset.classList.add('selected');
            } else {
                preset.classList.remove('selected');
            }
        });
    }

    updateBulkColorPresetSelection(color) {
        const colorPresets = document.querySelectorAll('#bulkColorPresets .color-preset');
        colorPresets.forEach(preset => {
            if (preset.dataset.color === color) {
                preset.classList.add('selected');
            } else {
                preset.classList.remove('selected');
            }
        });
    }

    saveTitleTemplate(entry) {
        this.titleTemplates[entry.title] = {
            description: entry.description,
            time: entry.time,
            category: entry.category,
            priority: entry.priority,
            color: entry.color
        };
        this.saveTitleTemplates();
        this.updateTitleSuggestions();
    }

    loadTitleTemplates() {
        try {
            const stored = localStorage.getItem('titleTemplates');
            return stored ? JSON.parse(stored) : {};
        } catch (e) {
            console.error('Error loading title templates:', e);
            return {};
        }
    }

    saveTitleTemplates() {
        try {
            localStorage.setItem('titleTemplates', JSON.stringify(this.titleTemplates));
        } catch (e) {
            console.error('Error saving title templates:', e);
        }
    }

    loadEntries() {
        try {
            const stored = localStorage.getItem('calendarEntries');
            return stored ? JSON.parse(stored) : {};
        } catch (e) {
            console.error('Error loading entries:', e);
            return {};
        }
    }

    saveEntries() {
        try {
            localStorage.setItem('calendarEntries', JSON.stringify(this.entries));
        } catch (e) {
            console.error('Error saving entries:', e);
            alert('Failed to save entries. Storage might be full.');
        }
    }
}

document.addEventListener('DOMContentLoaded', () => {
    new CalendarApp();
});
